#!/usr/bin/perl
# Copyright 2003 Bryan Ford
# Distributed under the GNU General Public License.
#
# Usage: mergedep <main-depfile> [<new-depfiles> ...]
#
# This script merges the contents of all <new-depfiles> specified
# on the command line into the single file <main-depfile>,
# which may or may not previously exist.
# Dependencies in the <new-depfiles> will override
# any existing dependencies for the same targets in <main-depfile>.
# The <new-depfiles> are deleted after <main-depfile> is updated.
#
# The <new-depfiles> are typically generated by GCC with the -MD option,
# and the <main-depfile> is typically included from a Makefile,
# as shown here for GNU 'make':
#
#	.deps: $(wildcard *.d)
#		perl mergedep $@ $^
#	-include .deps
#
# This script properly handles multiple dependencies per <new-depfile>,
# including dependencies having no target,
# so it is compatible with GCC3's -MP option.
#

sub readdeps {
	my $filename = shift;

	open(DEPFILE, $filename) or return 0;
	while (<DEPFILE>) {
		if (/([^:]*):([^\\:]*)([\\]?)$/) {
			my $target = $1;
			my $deplines = $2;
			my $slash = $3;
			while ($slash ne '') {
				$_ = <DEPFILE>;
				defined($_) or die
					"Unterminated dependency in $filename";
				/(^[ \t][^\\]*)([\\]?)$/ or die
					"Bad continuation line in $filename";
				$deplines = "$deplines\\\n$1";
				$slash = $2;
			}
			#print "DEPENDENCY [[$target]]: [[$deplines]]\n";
			$dephash{$target} = $deplines;
		} elsif (/^[#]?[ \t]*$/) {
			# ignore blank lines and comments
		} else {
			die "Bad dependency line in $filename: $_";
		}
	}
	close DEPFILE;
	return 1;
}


if ($#ARGV < 0) {
	print "Usage: mergedep <main-depfile> [<new-depfiles> ..]\n";
	exit(1);
}

%dephash = ();

# Read the main dependency file
$maindeps = $ARGV[0];
readdeps($maindeps);

# Read and merge in the new dependency files
foreach $i (1 .. $#ARGV) {
	readdeps($ARGV[$i]) or die "Can't open $ARGV[$i]";
}

# Update the main dependency file
open(DEPFILE, ">$maindeps.tmp") or die "Can't open output file $maindeps.tmp";
foreach $target (keys %dephash) {
	print DEPFILE "$target:$dephash{$target}";
}
close DEPFILE;
rename("$maindeps.tmp", "$maindeps") or die "Can't overwrite $maindeps";

# Finally, delete the new dependency files
foreach $i (1 .. $#ARGV) {
	unlink($ARGV[$i]) or print "Error removing $ARGV[$i]\n";
}

